//Wire library for I2C communication for debugging purposes
#include <Wire.h>

//Define control parameters. Variable Period of pulses "fPeriod" is in seconds.
//variable on_time controls the time the valve remains open. These values are automatically
//changed depending on the desired GUI value. 
float fPeriod = 30.0; //(seconds)
float on_time = 0.1*fPeriod; //% of fPeriod (seconds) (default value)
float fDutyCycle = 10.0;


//Variables used for timing 
unsigned long currentMillis, setupMillis, timer1 = 0, timer2 = 0, timer3 = 0;
//Boolean variables used for pulsing air and for determining if valves are closed
boolean Air_closed = false, pulse_started = false, N2_closed = false;
//Values for python and oxygen sensor also for storage of old values
float gui_value, sensor_value, previous_gui_value = 22.0;
//Variable for controls
int control_counter = 0;


/* 
Arduino Pinout:
  |   Arduino PIN    |   Motor Driver PIN   |         Purpose         - Color Wire   |
  |        A0        |         N/A          |       Photoreistor 1 N2   -  Brown     |
  |        A1        |         N/A          |       Photoresistor 2 Air -  White     |
  |        5         |         ENA          |       Speed Control Air -    Blue      |
  |        6         |         ENB          |       Speed Control N2  -   Violet     |
  |        7         |         IN1          |          Close Air      -   White      |
  |        8         |         IN2          |           Open Air      -   Brown      |
  |        9         |         IN3          |          Open N2       -   Yellow      |
  |       10         |         IN4          |           Close N2       -   Orange    |
  |       11         |         N/A          |           Pressure Relief              |

  //debugging (optional): hook another arduino up to I2C lines and ground to listen to I2C ouput
  //must load wire libraries example called "Slave Receiver" found in second example on:
  //https://www.arduino.cc/en/Tutorial/MasterWriter to second arduino to read data output
  |    This Arduino PIN      |   2nd Arduino Pins   |         Purpose           |
  |        A4 or SDA        |      A4 or SDA        |       SDA                 |
  |        A5 or SCL        |       A5 or SCL       |       SCL                 |
  |        Ground           |         Ground        |       Ground              |

*/
boolean output_debugging_messages = false;

int N2_Speed_PIN = 6;
int N2_Open_PIN = 9; 
int N2_Close_PIN = 10;
int Air_Speed_PIN = 5;
int Air_Open_PIN = 8;
int Air_Close_PIN = 7;
int N2_Sensor_PIN = 0;
int Air_Sensor_PIN = 1;
int pressure_PIN = 11;


//Code that runs once at beginning of program:
void setup() {
  
  if(output_debugging_messages)
    Wire.begin(); // join i2c bus for debugging purposes
     
  //begin serial communication with python
  Serial.begin(9600);

  //set the speed pins to output a pulse width modulated signal
  pinMode(N2_Speed_PIN, OUTPUT);
  pinMode(Air_Speed_PIN, OUTPUT);
  analogWrite(N2_Speed_PIN, 255); //0-255 180 is the slowest
  analogWrite(Air_Speed_PIN, 255); //0-255 180 is the slowest

  //set the control pins as outputs
  pinMode(N2_Open_PIN, OUTPUT);
  pinMode(N2_Close_PIN, OUTPUT);
  pinMode(Air_Open_PIN, OUTPUT);
  pinMode(Air_Close_PIN, OUTPUT);
  pinMode(pressure_PIN,OUTPUT);
    
  //open the motors all the way so that flowmeters can be adjusted
  digitalWrite(N2_Open_PIN, HIGH);
  digitalWrite(Air_Open_PIN, HIGH);
  delay(8000); //8 seconds
  digitalWrite(N2_Open_PIN, LOW);
  digitalWrite(Air_Open_PIN, LOW);

  //tell python Arduiono 1's while loop is beginning and wait for python to acknowledge
  //Arduino: "Arduino 1 Start"
  //Python: "ACK"
  boolean initial_transmission_not_sent = true;
  while(initial_transmission_not_sent){
    delay(1000);
    Serial.println("Arduino 1 Start");
    String initial_response = Serial.readString();
    Serial.println(initial_response);
    if(initial_response == "ACK"){
        initial_transmission_not_sent = false;
     }
  }
  //number of milliseconds elapsed since start of program to end of setup()
  setupMillis = millis();

  if(output_debugging_messages){
    //Tell debugging arduino we are done setting up
    Wire.beginTransmission(8); // transmit to device #8
    Wire.write("Done setting up ");        
    Wire.endTransmission();    // stop transmitting
  }
}




//Code that loops infinitely after setup() runs:
void loop() {
  //read in serial from python if available
  read_serial_COM();

  //get current time elapsed since start of void loop()
  currentMillis = millis()-setupMillis;

  //increment a control counter every 15 seconds
  if(currentMillis - timer3 >= 15000UL){
    control_counter++; //counter used to know how often to change duty cycle
    if(output_debugging_messages){
      Wire.beginTransmission(8); // transmit to device #8
      Wire.write("Control Counter: ");  
      Wire.write(control_counter);      
      Wire.endTransmission();    // stop transmitting
    }
    timer3 = currentMillis;
    if(control_counter > 100){
      control_counter = 0;
    }
  }

  float difference = gui_value - sensor_value;
  
  if(gui_value == 21 || difference > 0.2){
    open_valve('A');
    close_valve_all_the_way('N');
    digitalWrite(pressure_PIN,HIGH); //open pressure output
  }
  else if (gui_value == 0 || difference < -0.2){
    open_valve('N');
    close_valve_all_the_way('A');
    digitalWrite(pressure_PIN,HIGH); //open pressure output
  }
  else{
    close_valve_all_the_way('A');
    close_valve_all_the_way('N');
    digitalWrite(pressure_PIN,LOW); //close pressure output
  }
}



void read_serial_COM(){
  //Read serial communication from python and send acknowledge
  //Output from python will be "gui_value,sensor_value"
  //Also send back acknowledge so python knows arduino got it 
  if (Serial.available()) { //there are new values to read
    String myString = Serial.readString();
    gui_value = getValue(myString, ',', 0).toFloat();
    sensor_value = getValue(myString, ',', 1).toFloat();
    float temp_fDutyCycle = getValue(myString, ',', 2).toFloat();
    float temp_fPeriod = getValue(myString, ',', 3).toFloat();
    int override_control = getValue(myString, ',', 4).toInt();
    Serial.println("ACK");
      

    if(output_debugging_messages){
      //Tell Debugging Arduino
      Wire.beginTransmission(8); // transmit to device #8
      Wire.write("New GUI: ");  
      Wire.write(int(gui_value));      
      Wire.endTransmission();    // stop transmitting
      delay(300);
      Wire.beginTransmission(8); // transmit to device #8
      Wire.write("New Sensor: ");  
      Wire.write(int(sensor_value));      
      Wire.endTransmission();    // stop transmitting
      delay(300);
    }

    //calculate new on time if override controls are OFF
    if(override_control == 0){
      compute_on_time();
    }
    //calculate new on time if override controls are ON
    else if(fDutyCycle > 0.0 && fPeriod > 0.0){
      fDutyCycle = temp_fDutyCycle;
      fPeriod = temp_fPeriod;
      on_time = (fDutyCycle/100.0)*fPeriod;
    }
  }
}

//Function to parse a string and return an int 
String getValue(String data, char separator, int index)
{
    int found = 0;
    int strIndex[] = { 0, -1 };
    int maxIndex = data.length() - 1;
    for (int i = 0; i <= maxIndex && found <= index; i++) {
        if (data.charAt(i) == separator || i == maxIndex) {
            found++;
            strIndex[0] = strIndex[1] + 1;
            strIndex[1] = (i == maxIndex) ? i+1 : i;
        }
    }
    return found > index ? data.substring(strIndex[0], strIndex[1]) : "";
}

//Read value of photoresistors (for both air and N2) to determine whether or not the vavles are closed.
void read_photoresistors(){
  int photoresistor_1_value = analogRead(N2_Sensor_PIN);
  int photoresistor_2_value = analogRead(Air_Sensor_PIN);
  //For N2:
  if(photoresistor_1_value > 935) N2_closed = false; //Open
  else{
    N2_closed = true; //Closed
  }
  //For Air:
  if(photoresistor_2_value > 950) Air_closed = false; //Open
  else{
    Air_closed= true; //Closed
  }
}

void close_valve_all_the_way(char valve_select){
  int speed_PIN;
  int close_PIN;
  int open_PIN;
  boolean photoresistor_closed;
  if(valve_select == 'A'){
    speed_PIN = Air_Speed_PIN;
    close_PIN = Air_Close_PIN;
    open_PIN = Air_Open_PIN;
  }
  else if(valve_select == 'N'){
    speed_PIN = N2_Speed_PIN;
    close_PIN = N2_Close_PIN;
    open_PIN = N2_Open_PIN;
  }
  //set full speed
  analogWrite(speed_PIN, 255); //0-255 180 is the slowest
  //make sure valve isn't closing
  digitalWrite(open_PIN, LOW);
  //open
  digitalWrite(close_PIN, HIGH);
  
}

//Opens either vavlve at full speed
void open_valve(char valve_select){
  int speed_PIN;
  int close_PIN;
  int open_PIN;
  boolean photoresistor_closed;
  if(valve_select == 'A'){
    speed_PIN = Air_Speed_PIN;
    close_PIN = Air_Close_PIN;
    open_PIN = Air_Open_PIN;
  }
  else if(valve_select == 'N'){
    speed_PIN = N2_Speed_PIN;
    close_PIN = N2_Close_PIN;
    open_PIN = N2_Open_PIN;
  }
  //set full speed
  analogWrite(speed_PIN, 255); //0-255 180 is the slowest
  //make sure valve isn't closing
  digitalWrite(close_PIN, LOW);
  //open
  digitalWrite(open_PIN, HIGH);
}

//Closes either valve at the slowest speed. I did this because 
//from previous testing, I saw the black ball in the flowmeter has a delay
//after the valve closes because of pressure from the mixing chamber. 
void close_valve(char valve_select){
  int speed_PIN;
  int close_PIN;
  int open_PIN;
  boolean photoresistor_closed; 
  if(valve_select == 'A'){
    speed_PIN = Air_Speed_PIN;
    close_PIN = Air_Close_PIN;
    open_PIN = Air_Open_PIN;
  }
  else if(valve_select == 'N'){
    speed_PIN = N2_Speed_PIN;
    close_PIN = N2_Close_PIN;
    open_PIN = N2_Open_PIN;
  }
  //make sure valve isn't opening
  digitalWrite(open_PIN, LOW);
  //close slowly so black ball in flowmeter has time to drop down
  analogWrite(speed_PIN, 180); //0-255 180 is the slowest

  read_photoresistors();
  if(valve_select == 'A') photoresistor_closed = Air_closed;
  else if(valve_select == 'N') photoresistor_closed = N2_closed;
  //waits until photoresistor sensor goes to closed state
  while(!photoresistor_closed){ //while open
    digitalWrite(close_PIN, HIGH);
    delay(90);
    digitalWrite(close_PIN, LOW);
    delay(400);
    read_photoresistors();
    if(valve_select == 'A') photoresistor_closed = Air_closed;
    else if(valve_select == 'N') photoresistor_closed = N2_closed;  
  }
  //detected we are closed so stop closing
  digitalWrite(close_PIN, LOW);
  //set back to full speed
  analogWrite(speed_PIN, 255); //0-255 180 is the slowest
}

//Pulse in a certain amount of either air or nitrogen
//Ex: pulse_valve('A',10.1); (10.1 is in sec)
void pulse_valve(char valve_select, float on_time){
  int speed_PIN;
  int close_PIN;
  int open_PIN;
  boolean photoresistor_closed;
  if(valve_select == 'A'){
    speed_PIN = Air_Speed_PIN;
    close_PIN = Air_Close_PIN;
    open_PIN = Air_Open_PIN;
  }
  else if(valve_select == 'N'){
    speed_PIN = N2_Speed_PIN;
    close_PIN = N2_Close_PIN;
    open_PIN = N2_Open_PIN;
  }
  
  //Open slightly only every period
  if(pulse_started == false && (currentMillis - timer1 >= (unsigned long)(fPeriod*1000.0))){
    if(output_debugging_messages){
      Wire.beginTransmission(8); // transmit to device #8
      Wire.write("Time to open ");     
      Wire.endTransmission();    // stop transmitting
    }
    //restart the 1 minute timer1
    currentMillis = millis()-setupMillis;
    timer1 = currentMillis;
    pulse_started = true;
    read_photoresistors();
    if(valve_select == 'A') photoresistor_closed = Air_closed;
    else if(valve_select == 'N') photoresistor_closed = N2_closed;
    //open N2 slightly
    while(photoresistor_closed){  //while N2 closed
      digitalWrite(open_PIN, HIGH);
      read_photoresistors();
      if(valve_select == 'A') photoresistor_closed = Air_closed;
      else if(valve_select == 'N') photoresistor_closed = N2_closed;
    }
    //we are now slightly open so stop opening
    digitalWrite(open_PIN, LOW);
    currentMillis = millis()-setupMillis;
    timer2 = currentMillis;
  }

  //Close after "on_time" milliseconds
  if(pulse_started == true && (currentMillis - timer2 >= (unsigned long)(on_time*1000.0))){
    close_valve(valve_select);
    pulse_started = false;
    if(output_debugging_messages){
      Wire.beginTransmission(8); // transmit to device #8
      Wire.write("Time to close ");        
      Wire.endTransmission();    // stop transmitting
    }
  }
}

void compute_on_time(){
  
  //runs everytime there is a different gui value
  if(gui_value != previous_gui_value){
    switch(int(gui_value)){
    case 1:
      fPeriod = 30.0;
      fDutyCycle = 5.0;
      break;
    case 2:
      fPeriod = 30.0;
      fDutyCycle = 30.0;
      break;
    case 3:
      fPeriod = 30.0;
      fDutyCycle = 50.0;
      break;
    case 4:
      fPeriod = 30.0;
      fDutyCycle = 65.0;
      break;
    case 5:
      fPeriod = 30.0;
      fDutyCycle = 80.0;
      break;
    case 6:
      fPeriod = 30.0;
      fDutyCycle = 95.0;
      break;
    case 7: 
      fPeriod = 60.0;
      fDutyCycle = 70.0;
      break;
    case 8:
      fPeriod = 60.0;
      fDutyCycle = 80.0;
      break;
    case 9:
      fPeriod = 60.0;
      fDutyCycle = 90.0;
      break;
    case 10:
      fPeriod = 60.0;
      fDutyCycle = 95.0;
      break;
    case 11:
      fPeriod = 60.0;
      fDutyCycle = 95.0; //switched
      break;
    case 12:
      fPeriod = 60.0;
      fDutyCycle = 90.0;
      break;
    case 13:
      fPeriod = 60.0;
      fDutyCycle = 80.0;
      break;
    case 14:
      fPeriod = 30.0;
      fDutyCycle = 70.0;
      break;
    case 15:
      fPeriod = 30.0;
      fDutyCycle = 60.0;
      break;
    case 16:
      fPeriod = 30.0;
      fDutyCycle = 50.0;
      break;
    case 17:
      fPeriod = 30.0;
      fDutyCycle = 40.0;
      break;
    case 18:
      fPeriod = 30.0;
      fDutyCycle = 25.0;
      break;
    case 19:
      fPeriod = 30.0;
      fDutyCycle = 10.0;
      break;
    case 20:
      fPeriod = 30.0;
      fDutyCycle = 5.0;
      break;
    default:
      fPeriod = 30.0;
      fDutyCycle = 10.0;
    }
    control_counter = 0;
    previous_gui_value = gui_value;
  }
 
//  else{
//    float difference = gui_value - sensor_value;
//
//    //wait until we are in range after switching from normoxia to hypoxia or hypoxia to normoxia
//    if(abs(difference) < 6){    
//      if(difference >= 1 && control_counter > 3){
//        fDutyCycle += 6.0;
//        control_counter = 0;
//      }
//      else if(difference > 0.5 && difference <= 1 && control_counter > 3){
//        fDutyCycle += 4.0;
//        control_counter = 0;
//      }
//      else if(difference > 0.1 && difference <= 0.5 && control_counter > 4){
//        fDutyCycle += 2.0;
//        control_counter = 0;
//      }
//      else if(difference > -0.5 && difference <= -0.1 && control_counter > 4){
//        fDutyCycle -= 2.0;
//        control_counter = 0;
//      }
//      else if(difference > -1 && difference <= -0.5 && control_counter > 3){
//        fDutyCycle -= 4.0;
//        control_counter = 0;
//      }
//      else if(difference <= -1 && control_counter > 3){
//        fDutyCycle -= 6.0;
//        control_counter = 0;
//      }  
//    }
//  }

  if(fDutyCycle > 0.0 && fPeriod > 0.0){
    on_time = (fDutyCycle/100.0)*fPeriod;
  }

  if(output_debugging_messages){
      //Tell Debugging Arduino
      Wire.beginTransmission(8); // transmit to device #8
      Wire.write("Duty Cycle: ");  
      Wire.write(int(fDutyCycle));      
      Wire.endTransmission();    // stop transmitting
      delay(100);
      Wire.beginTransmission(8); // transmit to device #8
      Wire.write("ON time: ");  
      Wire.write(int(on_time));      
      Wire.endTransmission();    // stop transmitting
      delay(100);
      Wire.beginTransmission(8); // transmit to device #8
      Wire.write("Period: ");  
      Wire.write(int(fPeriod));      
      Wire.endTransmission();    // stop transmitting
      delay(100);
    }

    
}

